/*
** File:    UFLSamp1.c
** Author:  Rick Cameron
** Date:    8 Mar 93
**
** Purpose: Example user-defined functions.
*/

#if defined( _WIN32 ) && defined( _MSC_VER ) && _MSC_VER >= 1000
#define _export __declspec(dllexport)
#endif

#include <windows.h>
#include "ufdll.h"
#include "ufmain.h"
#include "ufuser.h"
#include "ufjob.h"

#include <ctype.h>

#define PicturePlaceholder 'x'

#define WildOneChar  '?'
#define WildAllChars '*'

#define SoundexLength 4

#define MAX_TIME_STRING_LEN     2 + 1 + 2 + 1 + 2 + 1 + 4 + 1


UFError FAR CR_EXPORT Now (UFParamBlock * ParamBlock);
UFError FAR CR_EXPORT Picture (UFParamBlock * ParamBlock);
UFError FAR CR_EXPORT LooksLike (UFParamBlock * ParamBlock);
UFError FAR CR_EXPORT Soundex (UFParamBlock * ParamBlock);

UFError FAR CR_EXPORT NowStringRetSize (UFMemCalcBlock * ParamBlock);
UFError FAR CR_EXPORT SoundexStringRetSize (UFMemCalcBlock * ParamBlock);

UF5FunctionDefStrings FunctionDefStrings [] =
{
    {"String Now", Now, NowStringRetSize, FALSE, FALSE},

    {"String Picture (String, String)", Picture, NULL, FALSE, FALSE},

    {"Boolean LooksLike (String, String)", LooksLike, NULL, FALSE, FALSE},

    {"String Soundex (String)", Soundex, SoundexStringRetSize, FALSE, FALSE},

    {NULL, NULL, NULL, FALSE, FALSE}
};

UFFunctionTemplates FunctionTemplates [] =
{
    {"Now!"},

    {"Picture (!, )"},

    {"LooksLike (!, )"},

    {"Soundex (!)"},

    {NULL}
};

UFFunctionExamples FunctionExamples [] =
{
    {"\tNow"},

    {"\tPicture (string, picture)"},

    {"\tLooksLike (string, mask)"},

    {"\tSoundex (string)"},

    {NULL}
};

char *ErrorTable [] =
{
    "no error",
};

void InitJob (struct JobInfo *jobInfo)
{
}

void TermJob (struct JobInfo *jobInfo)
{
}

UFError FAR CR_EXPORT Now (UFParamBlock * ParamBlock)
{
    // Get the current time and display it using the current locale settings.
    SYSTEMTIME systime;
    GetLocalTime (&systime);

    GetTimeFormat (LOCALE_USER_DEFAULT,
                                0,
                                &systime,
                                NULL,
                                ParamBlock->ReturnValue.ReturnString,
                                MAX_TIME_STRING_LEN);

    return UFNoError;
}

UFError FAR CR_EXPORT NowStringRetSize (UFMemCalcBlock * ParamBlock)
{
    // The longest string possible is "01:01:01 abcd"
    ParamBlock->UFMemCalcStringLength = MAX_TIME_STRING_LEN;
    return UFNoError;
}

static void copyUsingPicture (char *dest,
                              const char *source,
                              const char *picture)
{
    while (*picture != '\0')
    {
        if (tolower (*picture) == PicturePlaceholder)
            if (*source != '\0')
                *dest++ = *source++;
            else
                ; // don't insert anything
        else
            *dest++ = *picture;

        picture++;
    }

    // copy the rest of the source
    lstrcpy (dest, source);
}

UFError FAR CR_EXPORT Picture (UFParamBlock * ParamBlock)
{
    UFParamListElement *FirstParam,
                       *SecondParam;
    char *buffer;
    UFTInt32u outLength;

    FirstParam  = GetParam (ParamBlock, 1);
    SecondParam = GetParam (ParamBlock, 2);

    if (FirstParam == NULL || SecondParam == NULL)
        return UFNotEnoughParameters;

    // max length of return string is the sum of parameter string lengthes
    outLength = lstrlen (FirstParam->Parameter.ParamString) + lstrlen (FirstParam->Parameter.ParamString);
    if (outLength <= UFMaxStringLength)
        buffer = ParamBlock->ReturnValue.ReturnString;
    else
    {
        buffer = (char *) malloc (outLength + 1);
        if (buffer == 0)
            return UFNoMemory;
    }

    copyUsingPicture (buffer,
                      FirstParam->Parameter.ParamString,
                      SecondParam->Parameter.ParamString);

    if (outLength > UFMaxStringLength)
    {
        // trucate the output result to avoid overwriting memory
		// ParamBlock->ReturnValue.ReturnString is 255 bytes
		// UFMaxStringLength is 254
        lstrcpyn (ParamBlock->ReturnValue.ReturnString, buffer, UFMaxStringLength + 1);
        free (buffer);
    }
    return UFNoError;
}

static UFTBoolean looksLike (const char *string, const char *mask)
{
    while (*mask != '\0')
    {
        if (*mask == WildAllChars)
            return TRUE;
        else if (*mask == WildOneChar)
        {
            if (*string == '\0')
                return FALSE;
        }
        else if (*mask != *string)
            return FALSE;

        ++mask;
        ++string;
    }

    return (*string == '\0');
}

UFError FAR CR_EXPORT LooksLike (UFParamBlock * ParamBlock)
{
    UFParamListElement *FirstParam,
                       *SecondParam;

    FirstParam  = GetParam (ParamBlock, 1);
    SecondParam = GetParam (ParamBlock, 2);

    if (FirstParam == NULL || SecondParam == NULL)
        return UFNotEnoughParameters;

    ParamBlock->ReturnValue.ReturnBoolean
        = looksLike (FirstParam->Parameter.ParamString,
                     SecondParam->Parameter.ParamString);

    return UFNoError;
}

// static Int16u soundexCode

typedef enum
{
    nonAlpha = -1,
    ignored  = '0',
    bfpv     = '1',
    cgjkqsxz = '2',
    dt       = '3',
    l        = '4',
    mn       = '5',
    r        = '6'
} SoundexCode;

static char AnsiLowerChar (char c)
{
#ifdef _WIN32
	// for charactors that are a negative sign number e.g. '' 
	// because CharUpper and CharLower is expecting the hiword of LPTSTR 
	// to be 0 if the given parameter is a single char, otherwise the 
	// it will GPF.
	LPTSTR pc = (LPTSTR) (BYTE) c;
    return LOBYTE (LOWORD (CharLower (pc)));
#else
    return LOBYTE (LOWORD (AnsiLower (MAKELP (0, c))));
#endif
}

static char AnsiUpperChar (char c)
{
#ifdef _WIN32
	// for charactors that are a negative sign number e.g. '' 
	// because CharUpper and CharLower is expecting the hiword of LPTSTR 
	// to be 0 if the given parameter is a single char, otherwise the 
	// it will GPF.
	LPTSTR pc = (LPTSTR) (BYTE) c;  
    return LOBYTE (LOWORD (CharUpper (pc)));
#else
    return LOBYTE (LOWORD (AnsiUpper (MAKELP (0, c))));
#endif
}

static SoundexCode calcCode (char c)
{
    if (!IsCharAlpha (c))
        return nonAlpha;

    c = AnsiLowerChar (c);

    switch (c)
    {
        case 'b': case 'f': case 'p': case 'v':
             return bfpv;

        case 'c': case 'g': case 'j': case 'k':
        case 'q': case 's': case 'x': case 'z':
        case '': case '': case '': // s with an inverted ^
             return cgjkqsxz;

        case 'd': case 't':
        case '': case '':
             return dt;

        case 'l':
             return l;

        case 'm': case 'n':
        case '':
             return mn;

        case 'r':
             return r;

        default:
             return ignored;
    }
}

static const char *skipBlanks (const char *string)
{
    while (*string == ' ')
        ++string;

    return string;
}

static void calcSoundex (char *buffer, const char *string)
{
    const char *inCursor  = string,
               *bufferEnd = buffer + SoundexLength;
    char       *outCursor = buffer;
    SoundexCode code;

    inCursor = skipBlanks (inCursor);

    code = calcCode (*inCursor);

    if (code == nonAlpha)
        *outCursor++ = '0';
    else
        *outCursor++ = AnsiUpperChar (*inCursor++);

    while (code != nonAlpha
           && *inCursor != '\0'
           && outCursor < bufferEnd)
    {
        SoundexCode lastCode = code;

        code = calcCode (*inCursor);

        if (code == nonAlpha)
            ;
        else
        {
            if (code == lastCode)
                ;
            else if (code == ignored)
                code = lastCode;
            else
                *outCursor++ = code;

            ++inCursor;
        }
    }

    while (outCursor < bufferEnd)
           *outCursor++ = '0';

    *outCursor = '\0';
}

UFError FAR CR_EXPORT Soundex (UFParamBlock * ParamBlock)
{
    UFParamListElement * ParamPtr;

    ParamPtr = GetParam (ParamBlock, 1);
    if (ParamPtr == NULL)
        return UFNotEnoughParameters;

    calcSoundex (ParamBlock->ReturnValue.ReturnString,
                 ParamPtr->Parameter.ParamString);

    return UFNoError;
}

UFError FAR CR_EXPORT SoundexStringRetSize (UFMemCalcBlock *ParamBlock)
{
    ParamBlock->UFMemCalcStringLength = SoundexLength + 1;
    return UFNoError;
}
