#ifndef _LOG2_H_
#define _LOG2_H_

#include <assert.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <pthread.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>

template <class TYPE>
class Log2 {
public:

  Log2 (void)
    : _start (NULL),
      _actualLength (0),
      _perceivedLength (0),
      _isOpen (0) {}

  ~Log2 (void) {}

  enum { READ_WRITE = 0,
	 READ_ONLY = 1 };

  void open (const char * filename,
	     int readOnly = READ_ONLY)
  {
    if (!isOpen()) {

      // Save the filename.
      strncpy ((char *) _filename, (char *) filename, 255);
      
      int nobjs;
      
      // Open the file.
      // The file MUST exist and be non-empty.
      _fd = ::open (_filename, O_RDONLY, 00600);
      assert (_fd != -1);
      
      // Find out how long the file is.
      struct stat buf;
      fstat (_fd, &buf);
      
      // Calculate how many objects are in the file.
      nobjs = numberOfObjects((int) buf.st_size);
      assert (fileLength(nobjs) == (int) buf.st_size);

      _actualLength = nobjs;

      lseek(_fd, 0, SEEK_SET);
      read (_fd, (void *) &_perceivedLength, sizeof(int));
    }
  }


    // Close a log.
  inline void close (void) {
    ::close (_fd);
  }


  // Abort a log. (Don't write any changes.)
  inline void abort (void) {
    ::close (_fd);
  }

  // Access a log entry directly.
  inline TYPE& operator[] (int i) {
    TYPE t;
    lseek (_fd, i * sizeof(TYPE) + sizeof(int), SEEK_SET);
    read (_fd, (void *) &t, sizeof(TYPE));
    return t;
  }

  // Return the number of elements in the log.
  int length (void) {
    assert (_isOpen);
    int len = _perceivedLength;
    return len;
  }

private:

  int isOpen (void) {
    return _isOpen;
  }

  int isReadOnly (void) {
    return (_readOnlyMode == READ_ONLY);
  }

  // Given the total number of objects, return the file length.
  int fileLength (int nobjs) {
    return nobjs * sizeof(TYPE) + sizeof(int);
  }

  // Given the length of a file, return the total number of objects.
  int numberOfObjects (int fileLen) {
    return ((int) fileLen - sizeof(int)) / sizeof(TYPE);
  }

  char _filename[255];	// The log's filename.
  char * _start;	// The start address of the mmapped memory block.
  int _fd;
  int _isOpen;
  int _perceivedLength;
  int _actualLength;

};

#endif // _LOG2_H_
